<?php

namespace App\Helpers;

use App\Models\SystemSetting;

class CurrencyHelper
{
    /**
     * Currency symbols mapping
     */
    protected static array $symbols = [
        'USD' => '$',
        'EUR' => '€',
        'GBP' => '£',
        'NGN' => '₦',
        'GHS' => 'GH₵',
        'ZAR' => 'R',
        'KES' => 'KSh',
        'INR' => '₹',
        'JPY' => '¥',
        'KRW' => '₩',
        'CNY' => '¥',
        'CAD' => 'CA$',
        'AUD' => 'A$',
        'BRL' => 'R$',
        'MXN' => 'MX$',
        'CHF' => 'CHF',
        'SEK' => 'kr',
        'NOK' => 'kr',
        'DKK' => 'kr',
        'PLN' => 'zł',
        'THB' => '฿',
        'SGD' => 'S$',
        'HKD' => 'HK$',
        'MYR' => 'RM',
        'PHP' => '₱',
        'IDR' => 'Rp',
        'VND' => '₫',
        'AED' => 'د.إ',
        'SAR' => '﷼',
        'TRY' => '₺',
        'RUB' => '₽',
        'UAH' => '₴',
        'CZK' => 'Kč',
        'HUF' => 'Ft',
        'RON' => 'lei',
        'BGN' => 'лв',
        'HRK' => 'kn',
        'ILS' => '₪',
        'CLP' => 'CLP$',
        'COP' => 'COL$',
        'PEN' => 'S/',
        'ARS' => 'AR$',
    ];

    /**
     * Currencies with zero decimal places
     */
    protected static array $zeroDecimalCurrencies = [
        'BIF',
        'CLP',
        'DJF',
        'GNF',
        'JPY',
        'KMF',
        'KRW',
        'MGA',
        'PYG',
        'RWF',
        'UGX',
        'VND',
        'VUV',
        'XAF',
        'XOF',
        'XPF',
    ];

    /**
     * Get the system currency code
     */
    public static function getCode(): string
    {
        return SystemSetting::get('system_currency', 'USD');
    }

    /**
     * Get the currency symbol for a given code
     */
    public static function getSymbol(?string $code = null): string
    {
        $code = $code ?? self::getCode();

        return self::$symbols[$code] ?? $code;
    }

    /**
     * Format an amount with currency symbol
     * e.g., format(29.99) returns "$29.99" or "€29.99"
     */
    public static function format(float $amount, ?string $code = null, bool $showCode = false): string
    {
        $code = $code ?? self::getCode();
        $symbol = self::getSymbol($code);
        $decimals = self::getDecimalPlaces($code);

        $formatted = $symbol.number_format($amount, $decimals);

        if ($showCode) {
            $formatted .= ' '.$code;
        }

        return $formatted;
    }

    /**
     * Get decimal places for a currency
     */
    public static function getDecimalPlaces(string $code): int
    {
        return in_array($code, self::$zeroDecimalCurrencies) ? 0 : 2;
    }

    /**
     * Get all supported currencies for dropdown
     */
    public static function getSupportedCurrencies(): array
    {
        return array_keys(self::$symbols);
    }

    /**
     * Get currencies as options array for select inputs
     */
    public static function getCurrencyOptions(): array
    {
        $options = [];
        foreach (self::$symbols as $code => $symbol) {
            $options[$code] = "{$symbol} - {$code}";
        }

        return $options;
    }
}
